import random
from collections import defaultdict
from jelly_bean import JellyBean
import pytest

# Define the flavors and colors
FLAVORS = ["Lemon", "Pistachio", "Cantaloupe", "Pear", "Tangerine", "Apricot"]
COLORS = {
    "Lemon": "Yellow",
    "Pistachio": "Green",
    "Cantaloupe": "Orange",
    "Pear": "White",
    "Tangerine": "Orange",
    "Apricot": "Yellow"
}

def create_bag():
    """Create a bag with 24 candies, 4 per flavor."""
    return [(flavor, COLORS[flavor]) for flavor in FLAVORS for _ in range(4)]


def simulate_random_selection(bag, num_trials=100):
    flavor_coverage = []
    color_coverage = []
    total_flavors = set(FLAVORS)
    total_colors = set(COLORS.values())

    for _ in range(num_trials):
        selected_candies = random.sample(bag, 3)
        selected_flavors = {candy[0] for candy in selected_candies}
        selected_colors = {COLORS[candy[0]] for candy in selected_candies}

        flavor_coverage.append(len(selected_flavors) / len(total_flavors) * 100)
        color_coverage.append(len(selected_colors) / len(total_colors) * 100)

    avg_flavor_coverage = sum(flavor_coverage) / num_trials
    avg_color_coverage = sum(color_coverage) / num_trials

    return avg_flavor_coverage, avg_color_coverage


@pytest.mark.parametrize("trial", range(100))
def test_random_selection_coverage(trial):
    bag = create_bag()

    avg_flavor_coverage, avg_color_coverage = simulate_random_selection(bag, trial)

    print(f"Trial {trial + 1}:")
    print(f"  Flavor Coverage: {avg_flavor_coverage:.2f}%")
    print(f"  Color Coverage: {avg_color_coverage:.2f}%")
    
    assert 0 <= avg_flavor_coverage <= 100, "Flavor coverage out of bounds"
    assert 0 <= avg_color_coverage <= 100, "Color coverage out of bounds"

if __name__ == "__main__":
    pytest.main(["-s", "test_random_selection.py"])