import pytest
import matplotlib.pyplot as plt
import random
from guess_number_game import GuessNumberGame

def simulate_game():
    game = GuessNumberGame()
    while True:
        user_guess = random.randint(0, 9)  # Simulate user guess with random number for testing
        if game.guess(user_guess):
            return game.guess_count


@pytest.fixture(scope="module") # Integrate
def game_simulation_data():
    """Fixture to simulate multiple game runs and collect data."""
    num_simulations = 100  # Number of simulations to run
    guess_counts = []
    
    for _ in range(num_simulations):
        guess_count = simulate_game()
        guess_counts.append(guess_count)
    
    return guess_counts


@pytest.mark.dependency()
def test_simulation_1(game_simulation_data):
    guess_counts = game_simulation_data
    assert len(guess_counts) == 100
    assert all(1 <= count <= 100 for count in guess_counts)


@pytest.mark.dependency(depends=["test_simulation_1"])
def test_simulation_2(game_simulation_data):
    guess_counts = game_simulation_data
    avg_guesses = sum(guess_counts) / len(guess_counts)
    assert avg_guesses < 20


@pytest.mark.dependency(depends=["test_simulation_2"])
def test_game_simulation(game_simulation_data):
    guess_counts = game_simulation_data

    # Plot the guess count data
    plt.figure(figsize=(10, 5))
    plt.hist(guess_counts, bins=range(1, max(guess_counts) + 1), edgecolor='black')
    plt.xlabel('Number of Guesses')
    plt.ylabel('Frequency')
    plt.title('Distribution of Number of Guesses to Guess Correctly')
    plt.xticks(range(1, max(guess_counts) + 1))
    plt.grid(True)
    plt.savefig('guess_distribution.png')
    plt.show()

if __name__ == "__main__":
    pytest.main(["-v", "test_guess_number_game.py"])